﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Data.SqlClient;

namespace VIRP
{
    public sealed class DBBatchOperations
    {
        private static ConcurrentDictionary<Type, PropertyDescriptorCollection> KnownTypes;

        static DBBatchOperations()
        {
            KnownTypes = new ConcurrentDictionary<Type, PropertyDescriptorCollection>();
        }

        public void BulkInsert<T>(
            IEnumerable<T> collection,
            string connectionString,
            string destinationTableName,
            int batchSize = 50)
        {
            using (var connection = new SqlConnection(connectionString))
            {
                this.BulkInsert(
                    collection, 
                    connection, 
                    destinationTableName, 
                    batchSize);
            }
        }

        public void BulkInsert<T>(
            IEnumerable<T> collection,
            IDbConnection connection,
            string destinationTableName,
            int batchSize = 50)
        {
            var sqlConnection = connection as SqlConnection;
            if (sqlConnection == null)
                return;

            try
            {
                if (sqlConnection.State != ConnectionState.Open)
                    sqlConnection.Open();

                using (SqlTransaction transaction = sqlConnection.BeginTransaction())
                using (var bulkCopy = new SqlBulkCopy(sqlConnection, SqlBulkCopyOptions.Default, transaction))
                {
                    bulkCopy.BatchSize = batchSize;
                    bulkCopy.DestinationTableName = destinationTableName;
                    try
                    {
                        PropertyDescriptorCollection properties = GetAddType<T>();
                        foreach (PropertyDescriptor prop in properties)
                            bulkCopy.ColumnMappings.Add(prop.Name, prop.Name);

                        bulkCopy.WriteToServer(collection.AsDataTable());
                    }
                    catch (Exception)
                    {
                        transaction.Rollback();
                        throw;
                    }

                    transaction.Commit();
                }
            }
            finally
            {
                if (sqlConnection != null && sqlConnection.State == ConnectionState.Open)
                    sqlConnection.Close();
            } 
        }

        private static PropertyDescriptorCollection GetAddType<T>()
        {
            var key = typeof(T);

            return KnownTypes.GetOrAdd(key, properties => { return TypeDescriptor.GetProperties(key); });
        }
    }
}
